// LDA64_EthTest.cpp : This file contains the 'main' function. Program execution begins and ends there.
//
#include "stdafx.h"
using namespace std;
#include <stdio.h>
#include <stdint.h>         
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "lswdrvr.h"


const char* lswdevices[] = {"192.168.100.100"};
#define DEBUG_LEVEL 1
#define LSW_DEVICES  (sizeof(lswdevices)/sizeof(lswdevices[0]))

// IP Address Validator
#define DELIM "."

// --------------------------- Variables -------------------------------------------------
static char gldadevicesip[16];       // devices ip address
static int Swindex = 1;				// we just default to the Switch-1
static int Swout= SW_PORT1;


bool gbDeviceOpen = FALSE;
bool gbWantSaveSettings = FALSE;
bool gbWantSwitchStatus = FALSE;
bool gbWantSwitchCtrl = FALSE;
bool gbWantGetParam = FALSE;
bool gbQuietMode = FALSE;
bool gbBatchMode = FALSE;

//***************************************************************************
//
// Validate netconfig digits and return 1 if string contain only digits, else return 0
//
//*****************************************************************************
static int valid_digit(char *ip_str){
    while (*ip_str) {
        if (*ip_str >= '0' && *ip_str <= '9')
            ++ip_str;
        else
            return 0;
    }
    return 1;
}

//***************************************************************************
//
// Validate IP Configuration -return 1 if IP string is valid, else return 0
//
//*****************************************************************************
static int is_valid_ip(char *ip_str){
    int num, dots = 0;
    char *ptr;
    char lstr[16];
    char* next_token;

	strcpy_s(lstr,ip_str);

    if (lstr == NULL)
        return 0;

    ptr = strtok_s(lstr, DELIM, &next_token);

    if (ptr == NULL)
        return 0;

    while (ptr) {

        /* after parsing string, it must contain only digits */
        if (!valid_digit(ptr))
            return 0;

        num = atoi(ptr);

        /* check for valid IP */
        if (num >= 0 && num <= 255) {
            /* parse remaining string */
            ptr = strtok_s(NULL, DELIM, &next_token);
            if (ptr != NULL)
                ++dots;
        } else
            return 0;
    }

    /* valid IP string must contain 3 dots */
    if (dots != 3)
        return 0;
    return 1;
}

void PrintHelp()
{
	printf("Vaunix RF Switch Demonstration\n");
	printf("\n");
	printf("Hit CTRL+C to exit\n");
	printf("\n");

	printf(" --- Overall modes and device selection based on ip address ---\n");
	printf("  -d i 		Select the devices to work with, i is the device ip address (192.168.100.11)\n");
	printf("  -r  		Read the current information of the device\n");
	printf("  -y		Save the current settings in the device.\n");
	printf("\n");
	printf("  -b		Batch mode, exit immediately after sending commands to the Lab Bricks.\n");
	printf("  -q		Quiet mode, skip most outputs.\n");
	printf("\n");

	printf(" --- Commands to set/get rf switch parameters --- \n");
	printf("  -g  nn		Read the current RF Switch Port status of switch of the device, nn is switch index\n");
	printf("  -s  nn pp 	Set the current RF Switch Port of switch index of the device,  pp is rf switch output port, nn is switch index\n");
	printf("\n");
}

// -------------------- - MakeLower------------------------------

wchar_t MakeLowerW(wchar_t &wc)
{
	return wc = towlower(wc);
}	

// ParseCommandLine() will return FALSE to indicate that we received an invalid
// command or should abort for another reason.
bool ParseCommandLine(int argc, _TCHAR *argv[])
{
	char devipstr[16];
	size_t icount;

	enum {
		wantDash, wantGetParam, wantDevID, wantSwindex, wantSwindexSet, wantSWRFout
	} state = wantDash;

	for (int i = 1; i < argc; ++i) {
		// Convert each argument to lowercase
		wstring thisParam(argv[i]);
		for_each(thisParam.begin(), thisParam.end(), MakeLowerW);

		//printf("string:%ls\n",thisParam.c_str());

		if (state == wantDash)
		{
			if ('-' != thisParam[0])
			{
				printf("\n *** Error in command line syntax *** \n");
				PrintHelp();
				return FALSE;
			}
			// remove the dash from the front of the string
			thisParam = wstring(thisParam.begin() + 1, thisParam.end());

			// Identify the command line arguments
			if (L"d" == thisParam) {
				state = wantDevID;
			}
			else if (L"r" == thisParam) {
				gbWantGetParam = TRUE;
				state = wantDash;
			}
			else if (L"g" == thisParam) {
				state = wantSwindex;
			}
 			else if (L"s" == thisParam) {
				state = wantSwindexSet;
			}
			else if (L"y" == thisParam) {
				gbWantSaveSettings = TRUE;
				state = wantDash;
			}
			else if (L"b" == thisParam) {
				gbBatchMode = TRUE;
				state = wantDash;
			}
			else if (L"q" == thisParam) {
				gbQuietMode = TRUE;
				state = wantDash;
			}

			else {
				// this case is for "-?" and any argument we don't recognize
				PrintHelp();
				return FALSE;	// don't continue
			}
		}

		else {

			// save the whole substring and do conversions for each argument type

			switch (state){

			case wantDevID:
				wcstombs_s(&icount, devipstr, sizeof(devipstr), thisParam.c_str(), sizeof(devipstr));
				if(is_valid_ip(devipstr))
				{
					strcpy_s(gldadevicesip, devipstr);
//					printf("Device IP Address:%s\n",gldadevicesip);
					gbDeviceOpen = TRUE;
				}
				else
					printf("Invalid IP Address\n");
				state = wantDash;
				break;
 			case wantSwindex:
				Swindex = _wtoi(thisParam.c_str());
				gbWantSwitchStatus = TRUE;
				state = wantDash;
				break;

 			case wantSwindexSet:
				Swindex = _wtoi(thisParam.c_str());
				state = wantSWRFout;
				break;

 			case wantSWRFout:
				Swout = _wtoi(thisParam.c_str());
				gbWantSwitchCtrl = TRUE;
				state = wantDash;
				break;

			}
		}
	}

	if (state != wantDash) {
		// we are expecting an argument, if we didn't get one then print the help message
		PrintHelp();
		return FALSE;
	}

	// It's OK to continue
	return TRUE;
}


int _tmain(int argc, _TCHAR* argv[])
{
    bool realhardware;
    unsigned int index=0;
	int swindex;

	LSWDEVICE_DATA_T  lswdevicedata[LSW_DEVICES] = { 0 }; // MAX DEVICES list

	if (!ParseCommandLine(argc, argv))
		return 0;

	fnLSW_Init();
	/* If you have actual hardware attached, set this to TRUE. Setting to FALSE will run in test mode */
	realhardware = TRUE;
	fnLSW_SetTestMode(!realhardware);

	// Initialize the LDA Device
	if(gbDeviceOpen)
	{
	    if (fnLSW_InitDevice((char *)gldadevicesip))
	        printf("Device Connection Error:%s\n", gldadevicesip);
		else
			printf("Device Connection Success:%s\n", gldadevicesip);		
	}

	if(gbWantGetParam)
	{
		if (fnLSW_CheckDeviceReady((char *)gldadevicesip) == STATUS_OK)
        {
        	// Device IP
            printf("Device IP:%s\n", gldadevicesip);
            printf("----------------------------------------------\n");

			// Device Name
			fnLSW_GetModelName(gldadevicesip, lswdevicedata[index].modelname);
			printf("Model Name:%s\n", lswdevicedata[index].modelname);

			// Serial#
			fnLSW_GetSerialNumber(gldadevicesip, &lswdevicedata[index].serialnumber);
			printf("Serial Number:%d\n", lswdevicedata[index].serialnumber);

			// Sw Version #
			fnLSW_GetSoftwareVersion(gldadevicesip, lswdevicedata[index].swversion);
			printf("SW Version:%s\n", lswdevicedata[index].swversion);

			// IP Mode
			fnLSW_GetIPMode(gldadevicesip, &lswdevicedata[index].ipmode);
			printf("IP Mode:%d\n", lswdevicedata[index].ipmode);

			// IP Address
			fnLSW_GetIPAddress(gldadevicesip, lswdevicedata[index].ipaddress);
			printf("IP Address:%s\n", lswdevicedata[index].ipaddress);

			// Netmask
			fnLSW_GetNetmask(gldadevicesip, lswdevicedata[index].netmask);
			printf("Subnet Mask:%s\n", lswdevicedata[index].netmask);

			// Gateway
			fnLSW_GetGateway(gldadevicesip, lswdevicedata[index].gateway);
			printf("Gateway:%s\n", lswdevicedata[index].gateway);

			// Get max switches
			fnLSW_GetMaxSwitchDevices(gldadevicesip, &lswdevicedata[index].maxswitchdevices);
			printf("Max Switch Devices:%d\n", lswdevicedata[index].maxswitchdevices);

			// Switch RF output SW1
			for (swindex = 1; swindex <= lswdevicedata[index].maxswitchdevices; swindex++)
			{
				fnLSW_GetSwitchRFoutput(gldadevicesip, swindex, &lswdevicedata[index].rfswitchoutput[swindex - 1]);
				printf("SW%d-RF Output:%d\n", swindex, lswdevicedata[index].rfswitchoutput[swindex - 1]);
			}
        }
		gbWantGetParam = FALSE;
    }

	// Get Switch Status
	if (gbWantSwitchStatus) {
		if (!gbQuietMode)
			printf("Read RF Switch Port status of switch%d for device %s\n",
				Swindex, gldadevicesip);

		int resp = fnLSW_GetSwitchRFoutput(gldadevicesip, Swindex,
			&lswdevicedata[index].rfswitchoutput[Swindex - 1]);
		if (resp == STATUS_OK) {
			printf("SW%d-RF Output:%d\n", Swindex,
				lswdevicedata[index].rfswitchoutput[Swindex - 1]);
		}
		else {
			printf("Failed to get SW%d-RF Output\n", Swindex);
		}
	}


	// Set RF Switch port output
	if (gbWantSwitchCtrl) {
		if (!gbQuietMode)
			printf("Set RF Switch Port output of switch%d to %d for device %s\n",
				Swindex, Swout, gldadevicesip);
		int resp = fnLSW_SetSwitchRFoutput(gldadevicesip, Swindex, (LSW_SWPORT_T)Swout);
		if (resp == STATUS_OK) {
			printf("SW%d-RF Output set to %d\n", Swindex, Swout);
		}
		else {
			printf("Failed to set SW%d-RF Output\n", Swindex);
		}
	}


	// Save settings
	if (gbWantSaveSettings)
	{
		if (!gbQuietMode) printf("Saving the settings for device %s\n", gldadevicesip);
		fnLSW_SaveSettings((char *)gldadevicesip);
	}


	// Close the device socket at the end of the process
	if(gbDeviceOpen)
	{
		fnLSW_CloseDevice((char *)gldadevicesip);
		gbDeviceOpen = FALSE;

		// Clear out old device info
		memset(lswdevicedata, 0, sizeof(lswdevicedata));
	}

   
}

// Run program: Ctrl + F5 or Debug > Start Without Debugging menu
// Debug program: F5 or Debug > Start Debugging menu

// Tips for Getting Started: 
//   1. Use the Solution Explorer window to add/manage files
//   2. Use the Team Explorer window to connect to source control
//   3. Use the Output window to see build output and other messages
//   4. Use the Error List window to view errors
//   5. Go to Project > Add New Item to create new code files, or Project > Add Existing Item to add existing code files to the project
//   6. In the future, to open this project again, go to File > Open > Project and select the .sln file
