// -------------------------------------------------------------------------------------------------------------------------
// Module Name: LSW Ethernet Socket Communication Device Control Linux Library
// JA 02/03/2021    Initial Verison oF Ethernet Library
// NJB 08/20/2025   Updated to support LSW-802-PDT, LSW-802-P8T, LSW-203-PDT, LSW-403-PDT
// -------------------------------------------------------------------------------------------------------------------------
// This library uses .05db units for attenuation values, so 10db is represented by 200. (multiply by 20 to convert db to api units)
/// ---------- Include headers ----------------
#include <stdbool.h>
#include <stdint.h>       
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <time.h>
#include <errno.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <arpa/inet.h> 
#include <netdb.h>
#include <poll.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <errno.h>
#include <netinet/in.h>
#include <sys/timerfd.h>
#include <sys/wait.h> 
#include "lswsocket.h"

/// ---------- Macros ----------------
#define LSW_SOCKET_PORT  "40001"
#define FALSE 0
#define TRUE 1          
#define PACKET_CTRL_LEN 8
#define PACKET_INT_LEN  8

#define LIBVER "1.0.1"
#define LSW_DLLVERSION 0101        // we return an integer representation of the version with one byte each for major and minor version

// to force lsw library debugging messages choose the following definitions
#define DEBUG_OUT 1     /* set this to 1 in order to see debugging output, 2 for a ton of output, or 3 for many tons */

// IP Address Validator
#define DELIM "."

// Socket connection timeout
#define CONNECTION_TIMEOUT_SEC 5

// Thread States Macros
#define THREAD_IDLE 0
#define THREAD_START 1
#define THREAD_EXIT 3
#define THREAD_DEAD 4
#define THREAD_ERROR -1

/* -----------------------------------------------------------------
Global Data to maintain the Devices data & states for each process
-------------------------------------------------------------------*/
bool TestMode = FALSE;          // if TestMode is true we fake it -- no HW access
int  lswdevcount = 0;
time_t starttime;
LSWPARAMS lsw [MAXDEVICES];      // an array of structures each of which holds the info for a given device

/* stuff for the threads */
pthread_t threads[MAXDEVICES];


/// ---------- Const Definitions----------
// product names
const char *sVNX_devicenames[] ={"LSW-802P4T", "LSW-802PDT", "LSW-802P8T", "LSW-203PDT", "LSW-403PDT"};

char STATUS_QUERY_ID[]= {VNX_GETSERNUM, VNX_MODELNAME, VNX_SWVERSION, VNX_IPMODE, VNX_IPADDR, VNX_IPMASK,
                         VNX_IPGW, VNX_MAXPORTS};

char LibVersion[] = LIBVER;

static void *rxdatahandler_callback (void *ptr);

/// ----------  Static Fucntion Calls ----------------
//***************************************************************************
//
// Function call for returning the LibVersion
//
//*****************************************************************************
char* fnLSW_LibVersion(void) {
    return LibVersion;
}

//***************************************************************************
//
// Function call for Settig into test mode
//
//*****************************************************************************
void fnLSW_SetTestMode(bool testmode) {
    TestMode = testmode;
}


//***************************************************************************
//
// Function call for delay loop
//
//*****************************************************************************
/* usleep is deprecated, so this is a function using nanosleep() */
void catnap(long naptime) {
  // naptime comes in as the number of ms we want. 20 = 20ms
  struct timespec sleepytime;
   sleepytime.tv_sec = 0;
   sleepytime.tv_nsec = naptime * 1000000L;

   nanosleep(&sleepytime , NULL);
}


//***************************************************************************
//
// Validate netconfig digits and return 1 if string contain only digits, else return 0
//
//*****************************************************************************
static int valid_digit(char *ip_str){
    while (*ip_str) {
        if (*ip_str >= '0' && *ip_str <= '9')
            ++ip_str;
        else
            return 0;
    }
    return 1;
}

//***************************************************************************
//
// Validate IP Configuration -return 1 if IP string is valid, else return 0
//
//*****************************************************************************
static int is_valid_ip(char *ip_str){
    int num, dots = 0;
    char *ptr;
    char lstr[16];
    strcpy(lstr,ip_str);

    if (lstr == NULL)
        return 0;

    ptr = strtok(lstr, DELIM);

    if (ptr == NULL)
        return 0;

    while (ptr) {

        /* after parsing string, it must contain only digits */
        if (!valid_digit(ptr))
            return 0;

        num = atoi(ptr);

        /* check for valid IP */
        if (num >= 0 && num <= 255) {
            /* parse remaining string */
            ptr = strtok(NULL, DELIM);
            if (ptr != NULL)
                ++dots;
        } else
            return 0;
    }

    /* valid IP string must contain 3 dots */
    if (dots != 3)
        return 0;
    return 1;
}

//***************************************************************************
//
// Function call to Get DeviceID based on the deviceip from the device entry list
//
//*****************************************************************************
static int GetDeviceID(char* deviceip){
    int index=-1;

    // Loop through to get the device index
    for(index=0; index < MAXDEVICES; index++)
    {
        if(strcmp(lsw[index].deviceip, deviceip)==0)
            return index;
    }

    return index;
}

//***************************************************************************
//
// Function call to check whether lsw device socket open or not
//
//*****************************************************************************
static bool isDeviceSockOpen(char* deviceip) {
  int devid;
  if (TestMode) return TRUE;// in test mode all devices are always available
  devid = GetDeviceID(deviceip);
//  printf("devid:%d\n",devid);
  if(devid >=0)
  {
//      printf("devicesockfd:%d\n",lsw[devid].devicesockfd);
      // Even for multichannel devices, we can use channel 0 to know if the device is open
      if (lsw[devid].devicesockfd != 0)
        return TRUE;
      else
        return FALSE;
  }
  else
    return FALSE;
}
//***************************************************************************
//
// Function call to Open the Device Socket Connection Open
//
//*****************************************************************************
static int vnxDeviceOpen(char* deviceip) {
    int sockfd = -1;
    struct addrinfo hints;
    struct addrinfo *result, *rp;
    int flags;
    int iResult;

    memset(&hints, 0, sizeof(struct addrinfo));
    hints.ai_canonname = NULL;
    hints.ai_addr = NULL;
    hints.ai_next = NULL;
    hints.ai_family = AF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_flags = AI_NUMERICSERV;

//    printf("IP:%s\n",deviceip);
    printf("Opening Device Socket Connection:%s\n", deviceip);
 
    // Convert IPv4 and IPv6 addresses from text to binary form 
    if(inet_pton(AF_INET, deviceip, &hints.ai_addr)<=0)  
    { 
        if (DEBUG_OUT > 1) printf("\nInvalid address/ Address not supported \n"); 
        return -1; 
    } 
    
    // Check Device is already Open or not
    if (getaddrinfo(deviceip, LSW_SOCKET_PORT, &hints, &result) != 0)
        return -1;

    for (rp = result; rp != NULL; rp = rp->ai_next)
    {
        //  Open the Socket Connection 
        sockfd = socket(rp->ai_family, rp->ai_socktype, rp->ai_protocol);
        if (sockfd == -1)
            continue;

        // Set socket to non-blocking
        flags = fcntl(sockfd, F_GETFL, 0);
        if (flags == -1) {
            close(sockfd);
            continue;
        }
        if (fcntl(sockfd, F_SETFL, flags | O_NONBLOCK) == -1) {
            close(sockfd);
            continue;
        }

        iResult = connect(sockfd, rp->ai_addr, rp->ai_addrlen);
        if (iResult < 0) {
            if (errno != EINPROGRESS) {
                close(sockfd);
                sockfd = -1;
                continue;
            }
        }

        if (iResult < 0) {
            fd_set writefds;
            FD_ZERO(&writefds);
            FD_SET(sockfd, &writefds);

            struct timeval tv;
            tv.tv_sec = CONNECTION_TIMEOUT_SEC;
            tv.tv_usec = 0;

            iResult = select(sockfd + 1, NULL, &writefds, NULL, &tv);
            if (iResult <= 0) {
                close(sockfd);
                sockfd = -1;
                continue;
            }

            // Check if the connection actually succeeded
            int optVal;
            socklen_t optLen = sizeof(optVal);
            if (getsockopt(sockfd, SOL_SOCKET, SO_ERROR, &optVal, &optLen) < 0 || optVal != 0) {
                close(sockfd);
                sockfd = -1;
                continue;
            }
        }

        // Restore blocking mode
        if (fcntl(sockfd, F_SETFL, flags & ~O_NONBLOCK) == -1) {
            close(sockfd);
            sockfd = -1;
            continue;
        }

        //Successfully connected
        break;
    }

    if (rp == NULL)
    {
        if (sockfd != -1) close(sockfd);
        return -1;
    }

    freeaddrinfo(result);

    return sockfd;
}

//***************************************************************************
//
// Function call to Send the VNX commands through the Socket to the given ip
//  device
//*****************************************************************************
static bool SendReport(int deviceID, char command, char *pBuffer, int cbBuffer)
{
  int index;
  int timedout;

  if (TestMode) {
    return TRUE;
  }

  // Make sure the buffer that is being passed to us fits
  if (cbBuffer > HR_BLOCKSIZE) {
    // Report too big, don't send!
    return FALSE;
  }

  // lets make sure our command doesn't have any junk bits in it
  for (index=0; index<HID_REPORT_LENGTH; index++)
      lsw[deviceID].sndbuff[index] = 0;

  if (DEBUG_OUT > 1) printf("SR: command=%x cbBuffer=%d\r\n", (uint8_t)command, cbBuffer);
  lsw[deviceID].sndbuff[0] = command;       // command to device
  lsw[deviceID].sndbuff[1] = cbBuffer;
  for (index=0; index<cbBuffer; index++)
    lsw[deviceID].sndbuff[index+2] = pBuffer[index];

  if (DEBUG_OUT > 1) {
    printf("SR to device %d: ", deviceID);
    for (index=0; index<8; index++) {
      printf("%02x ", (uint8_t)lsw[deviceID].sndbuff[index]);
    }
    printf("\r\n");
  }

  lsw[deviceID].responseready = 0; // Clear Device Response before command send

  // Call Socket Function to send packet data to the device
  if (write(lsw[deviceID].devicesockfd, lsw[deviceID].sndbuff, PACKET_CTRL_LEN) < 0 )
        if (DEBUG_OUT > 1) printf("sending Message Error!");

  //Check for Get /Set Message type and expect response if Get message only
  if((command &  VNX_SET) == 0)
  {
	  // --- then we wait for the read thread's parser to signal that it got the response
	  starttime = time(NULL);
	  timedout = 0;
	  
	  // wait until the value is decoded or 1 seconds have gone by
	  // modified to yield the thread during the wait
	  while ((!lsw[deviceID].responseready) && (0 == timedout)) {
	    catnap(10);   // yield for 10 ms, speed matters here
	    if ((time(NULL)-starttime) > 1) timedout = 1;
	  }

	  return (0 == timedout) ? TRUE : FALSE;
  }
  else
  	return TRUE;
}

/*******************************************************************************
 * \brief    Find Position of the bit set for channel mask identification
 *
 ******************************************************************************/
 // A utility function to check whether n is a power of 2 or not. 
static int isPowerOfTwo(unsigned n) 
{ 
    return n && (!(n & (n - 1))); 
} 
static int findPosition(unsigned n) 
{ 
    if (!isPowerOfTwo(n)) 
        return -1; 
  
    unsigned i = 1, pos = 1; 
  
    // Iterate through bits of n till we find a set bit 
    // i&n will be non-zero only when 'i' and 'n' have a set bit 
    // at same position 
    while (!(i & n)) { 
        // Unset current bit and set the next bit in 'i' 
        i = i << 1; 
  
        // increment position 
        ++pos; 
    } 
  
    return pos; 
} 


/*******************************************************************************
 * \brief    Parse Data Frame received from the socket
 *
 ******************************************************************************/
static int parseDataFrame(char* msg, int count, int sockfd, int devid)
{
    int index;
    RESPONSE_DATA_T dataresp;
    unsigned short dataval_16;
    unsigned int dataval_32;
    unsigned char dataval_8;
	char *dataptr;
	unsigned char swindex;

#if 0
    for(index=0; index < count; index++)
    {
        printf("%02X",(unsigned char)msg[index]);
    }

    printf("Parse Data Frame:(Sockfd-%d, Devid-%d, Count-%d)\n",sockfd,devid,count);
#endif

    dataresp.command = msg[0]; // Command ID
    dataresp.length = msg[1];  // Data Length
    dataptr = &msg[2]; // Data buffer Start
    dataval_16 = *(unsigned short*)&msg[2];   // Data of 16 bits
    dataval_32 = *(unsigned int*)&msg[2]; // Data of 32 bits
    dataval_8 =  *(unsigned char*)&msg[2]; // Data of 8bits 
    swindex = *(unsigned char*)&msg[6]; // Offset of 4th byte

    switch(dataresp.command){

        case VNX_GETSERNUM:
            lsw[devid].SerialNumber = dataval_32;
            break;

        case VNX_MODELNAME:
            if (strstr((char*)&msg[2], "802PDT") || strstr((char*)&msg[2], "02-PDT"))
            {
                strncpy(lsw[devid].ModelName, "LSW-802PDT", sizeof(lsw[devid].ModelName) - 1);
                lsw[devid].ModelName[sizeof(lsw[devid].ModelName) - 1] = '\0';
                lsw[devid].NumSwitches = 2;
            }
            else if (strstr((char*)&msg[2], "802P4T") || strstr((char*)&msg[2], "02-P4T"))
            {
                strncpy(lsw[devid].ModelName, "LSW-802P4T", sizeof(lsw[devid].ModelName) - 1);
                lsw[devid].ModelName[sizeof(lsw[devid].ModelName) - 1] = '\0';
                lsw[devid].NumSwitches = 4;
            }
            else if (strstr((char*)&msg[2], "802P8T") || strstr((char*)&msg[2], "02-P8T"))
            {
                strncpy(lsw[devid].ModelName, "LSW-802P8T", sizeof(lsw[devid].ModelName) - 1);
                lsw[devid].ModelName[sizeof(lsw[devid].ModelName) - 1] = '\0';
                lsw[devid].NumSwitches = 8;
            }
            else if (strstr((char*)&msg[2], "203PDT") || strstr((char*)&msg[2], "03-PDT"))
            {
                strncpy(lsw[devid].ModelName, "LSW-203PDT", sizeof(lsw[devid].ModelName) - 1);
                lsw[devid].ModelName[sizeof(lsw[devid].ModelName) - 1] = '\0';
                lsw[devid].NumSwitches = 2;
            }
            else if (strstr((char*)&msg[2], "403PDT"))
            {
                strncpy(lsw[devid].ModelName, "LSW-403PDT", sizeof(lsw[devid].ModelName) - 1);
                lsw[devid].ModelName[sizeof(lsw[devid].ModelName) - 1] = '\0';
                lsw[devid].NumSwitches = 2;
            }
            else
            {
                // copy the model name from the incoming message
                size_t copylen = (dataresp.length < sizeof(lsw[devid].ModelName) - 1)
                                ? dataresp.length
                                : sizeof(lsw[devid].ModelName) - 1;
                memcpy(lsw[devid].ModelName, &msg[2], copylen);
                lsw[devid].ModelName[copylen] = '\0';
                lsw[devid].NumSwitches = 2;
            }
            break;

        case VNX_SWVERSION:
            strncpy (lsw[devid].Swversion, &msg[2],dataresp.length);  // save the device's model name
            break;
            
        case VNX_IPMODE:
            lsw[devid].ipmode = (int)dataval_8;
            break;

        case VNX_IPADDR:
            sprintf(lsw[devid].ipaddress, "%d.%d.%d.%d", (dataval_32 >> 24) & 0xff, (dataval_32 >> 16) & 0xff,
                 (dataval_32 >> 8) & 0xff, dataval_32 & 0xff);
            break;

        case VNX_IPMASK:
            sprintf(lsw[devid].netmask, "%d.%d.%d.%d", (dataval_32 >> 24) & 0xff, (dataval_32 >> 16) & 0xff,
                 (dataval_32 >> 8) & 0xff, dataval_32 & 0xff);
            break;

        case VNX_IPGW:
            sprintf(lsw[devid].gateway, "%d.%d.%d.%d", (dataval_32 >> 24) & 0xff, (dataval_32 >> 16) & 0xff,
                 (dataval_32 >> 8) & 0xff, dataval_32 & 0xff);
            break;

		case VNX_MAXPORTS:
			lsw[devid].Maxswports = (int)dataval_16;
			break;

        case VNX_SWSELECT:
            lsw[devid].rfswitchoutput[swindex-1] = dataval_32;
			lsw[devid].deviceready = 1; // Set Device Ready after init calls
            break;

        default:
            break;
    }

    lsw[devid].responseready = 1;  // Device Response Received

    return STATUS_OK;
}

/*******************************************************************************
 * \brief    Reads Data from the socket
 ******************************************************************************/
static int readdatafromSock(int sockfd, int devid)
{
    int count;
    char *buf_pr;
    char buf_st[512] = {0};
    int index;

    buf_pr = buf_st;
        
    if ((count = read(sockfd, buf_pr, 512)) >= 0) {

        //printf("READ FROM SOCKET(%d): Count-%d\n", sockfd, count);
        // Check Socket data response > 8 then split into 8 bytes chunks and parse the command data
        if(count > HID_REPORT_LENGTH)
        {
            for(index=0; index < count; index+=8)
            {
                //printf("Send Frame Data(%d,%d)\n",index, count);
                parseDataFrame(buf_pr+index, 8, sockfd, devid);
            }
        }
        else
            parseDataFrame(buf_pr, count, sockfd, devid);
    }
    return 0;
}

/*******************************************************************************
 * \brief    Callback for Receiver data handler of thread
 ******************************************************************************/
static void *rxdatahandler_callback (void *threadID) {
  int tid, ret = 0;;
  struct pollfd poll_fd[1];
  tid = (int)(uintptr_t)threadID;
//  printf("Thread Callback:%d\n",tid);
  for (;;) {
     //TODO: rx messages from CODU server and wait for terminate command
      poll_fd[0].fd = lsw[tid].devicesockfd;
      poll_fd[0].events = POLLIN | POLLRDNORM;
      poll_fd[0].revents = 0;
            
      for(;;)
      {
            ret = poll(poll_fd, 1, -1);
            if (ret > 0)
            {
                if (poll_fd[0].revents & POLLIN || poll_fd[0].revents & POLLRDNORM)
                {
                    poll_fd[0].revents = 0;
                    // Read data
                    readdatafromSock(lsw[tid].devicesockfd, tid);
                }
            }    
        }
    }
    return 0;
}

//***************************************************************************
//
// Function call to Get the parameter from the device using message
//  
//*****************************************************************************
static bool GetParameter(char* deviceip, int GetParam)
{
    char VNX_param[6] = {0, 0, 0, 0, 0, 0};
    int devid=0;

    // Don't talk to the hardware in test mode
    if (TestMode) {
        return STATUS_OK;
    }

    devid = GetDeviceID(deviceip);

    if( devid >=0)
    {
        // Check Device open or not
        if (!isDeviceSockOpen(deviceip))
            return STATUS_ERROR;

        if (!SendReport(devid, GetParam, VNX_param, 0)) {
          return STATUS_ERROR;
        }       
    }
    else 
        return STATUS_ERROR;
    return  STATUS_OK;
}

//***************************************************************************
//
// Function call to Get the parameter from the device using message
//  
//*****************************************************************************
static bool GetParameterWithIndex(char* deviceip, int GetParam, int index)
{
    char VNX_param[6] = {0, 0, 0, 0, 0, 0};
    int devid=0;

    // Don't talk to the hardware in test mode
    if (TestMode) {
        return STATUS_OK;
    }

    devid = GetDeviceID(deviceip);

    if( devid >=0)
    {
        // Check Device open or not
        if (!isDeviceSockOpen(deviceip))
            return STATUS_ERROR;

			VNX_param[4] = index;
        if (!SendReport(devid, GetParam, VNX_param, 5)) {
          return STATUS_ERROR;
        }       
    }
    else 
        return STATUS_ERROR;
    return  STATUS_OK;
}


//***************************************************************************
//
// Function call to Initialize all the  device data structures with default data
//  
//*****************************************************************************
void fnLSW_Init(void) {
    /* clear out the storage structure. Must be called once before anything else */
    int index;

	for(index=0; index<MAXDEVICES; index++)
	{
	    lsw[index].SerialNumber = 0;        // clear the serial number
	    lsw[index].ModelName[0] = 0;        // put a null string in each model name field
	    lsw[index].devicesockfd = 0;        // clear the device handles
	    lsw[index].deviceready = 0;
	    lsw[index].responseready = 0;       // Device Response ready status  
	    strcpy(lsw[index].deviceip, "");
	}
    if (DEBUG_OUT > 0)  printf("Linux SDK version %s\r\n", fnLSW_LibVersion());
}

//***************************************************************************
//
// Function call to Initialize the LSW Device and Open the socket connection
// to that device for communication 
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_InitDevice(char* deviceip) {
    int sockfd;
    int devid, err;
    unsigned int index;

    // Check whether we reached max device count
    if ((lswdevcount >= MAXDEVICES)) {
        return STATUS_ERROR;
    }

    //check device ip format 
    if(!is_valid_ip(deviceip))
        return STATUS_ERROR;

    // Check whether already the socket open for that device
//  if(isDeviceSockOpen(deviceip))
//      return STATUS_ERROR;

    if (TestMode){
        int duplicate = 0;

        // Check existing devices for duplicate IP
        for (int i = 0; i < lswdevcount; i++) {
            if (strcmp(lsw[i].deviceip, deviceip) == 0) {
                duplicate = 1;
                break;
            }
        }

        if (!duplicate) {
            strcpy(lsw[lswdevcount].deviceip, deviceip);

            lsw[lswdevcount].SerialNumber = 12345 + lswdevcount;
            lsw[lswdevcount].devicesockfd = 1234;
            lsw[lswdevcount].deviceready = 1;
            lsw[lswdevcount].responseready = 1;
            strcpy(lsw[lswdevcount].ModelName, "LSW-802P4T");
            lsw[lswdevcount].ipmode = 0;
            strcpy(lsw[lswdevcount].ipaddress, deviceip);
            strcpy(lsw[lswdevcount].netmask, "255.255.255.0");
            strcpy(lsw[lswdevcount].gateway, "192.168.1.1");
            strcpy(lsw[lswdevcount].Swversion, "0.0.0");
            lsw[lswdevcount].Maxswports = 4;
            lsw[lswdevcount].NumSwitches = 4;
            lsw[lswdevcount].rfswitchoutput[0] = 1;

            lswdevcount++;
        } else {
            return STATUS_OK; // Duplicate device IP found in test mode
        }
    }
    else {
        // Copy the device IP to the data structure
        strcpy(lsw[lswdevcount].deviceip, deviceip);

        // Go ahead and open a handle to the hardware
        sockfd = vnxDeviceOpen(deviceip);
        if (sockfd == -1)  // vnxDeviceOpen returns 0 (STATUS_OK) if the open succeeded, otherwise an error code
          return STATUS_ERROR;

        lsw[lswdevcount].devicesockfd = sockfd;
        lswdevcount++;

//        printf("Socketfd:%d\n",sockfd);
        devid = GetDeviceID(deviceip);
        // Create a device threads for reading the data from the device using sockets
        err = pthread_create(&threads[devid], NULL, rxdatahandler_callback, (void*)(uintptr_t)devid);

        // Get the current status details of the device parameter
        for(index=0; index < sizeof(STATUS_QUERY_ID); index++)
        {
            GetParameter(deviceip, STATUS_QUERY_ID[index]);
        }

		// Read all the switch devices status
		for(index=1; index <= MAX_SWDEVICES; index++)
		{
			GetParameterWithIndex(deviceip, VNX_SWSELECT, index);
		}

        // Wait for device to be ready before returing from init call
        while(fnLSW_CheckDeviceReady(deviceip));
    } // end of real device open process case

    // if we got here everything worked OK
    return STATUS_OK;
}


//***************************************************************************
//
// Function call to close the socket of the device open
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_CloseDevice(char* deviceip) {
    int sockfd, devid;
        
    //check device ip format 
    if(!is_valid_ip(deviceip))
      return STATUS_ERROR;

    // Check Device open
    if(isDeviceSockOpen(deviceip))
    {
        devid = GetDeviceID(deviceip);
        // If Device exists then read the data 
        if(devid >= 0)
        {
            strcpy(lsw[devid].deviceip,"");
            lswdevcount--;
            if (!TestMode){
                sockfd = lsw[devid].devicesockfd;
                close(sockfd);
                shutdown(sockfd,0);
                shutdown(sockfd,1);
                shutdown(sockfd,2);
            }
            lsw[devid].devicesockfd = 0;
        }
        else
            return STATUS_ERROR;
    }
    else
        return STATUS_ERROR;
  return STATUS_OK;

}


//***************************************************************************
//
// Function call to Check Device is Ready or not
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_CheckDeviceReady(char* deviceip ) {
    int devid=0;
    if (!isDeviceSockOpen(deviceip))
        return STATUS_ERROR;

    if (TestMode) {
        return STATUS_OK; // in test mode we always return OK
    }

    devid = GetDeviceID(deviceip);


    // If Device exists then read the data 
    if((devid >= 0) && (lsw[devid].deviceready !=0))
    {
        return STATUS_OK;
    }
    else
        return STATUS_ERROR;
}

//***************************************************************************
//
// Function call to Get Model name from the device
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_GetModelName(char* deviceip, char *ModelName) {
    int devid=0;
    if (!isDeviceSockOpen(deviceip))
        return STATUS_ERROR;

    devid = GetDeviceID(deviceip);

    // If Device exists then read the data 
    if(devid >= 0)
    {
        GetParameter(deviceip, VNX_MODELNAME);
        strcpy(ModelName, lsw[devid].ModelName);
    }
    else
        return STATUS_ERROR;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get Serial# from the device
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_GetSerialNumber(char* deviceip, int* respdata){
    int devid=0;
    if (!isDeviceSockOpen(deviceip))
        return STATUS_ERROR;

    devid = GetDeviceID(deviceip);

    // If Device exists then read the data 
    if(devid >= 0)
    {
        GetParameter(deviceip, VNX_GETSERNUM);
        *respdata = lsw[devid].SerialNumber;
    }
    else
        return STATUS_ERROR;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get Software Version from the device
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_GetSoftwareVersion(char* deviceip, char* respdata){
    int devid=0;
    if (!isDeviceSockOpen(deviceip))
        return STATUS_ERROR;

    devid = GetDeviceID(deviceip);

    // If Device exists then read the data 
    if(devid >= 0)
    {
        GetParameter(deviceip, VNX_SWVERSION);
        strcpy(respdata, lsw[devid].Swversion);
    }
    else
        return STATUS_ERROR;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get IP Mode from the device
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_GetIPMode(char* deviceip, int* respdata){
    int devid=0;
    if (!isDeviceSockOpen(deviceip))
        return STATUS_ERROR;

    devid = GetDeviceID(deviceip);

    // If Device exists then read the data 
    if(devid >= 0)
    {
        GetParameter(deviceip, VNX_IPMODE);
        *respdata = lsw[devid].ipmode;
    }
    else
        return STATUS_ERROR;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get IP Address from the device
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_GetIPAddress(char* deviceip, char* respdata){
    int devid=0;
    if (!isDeviceSockOpen(deviceip))
        return STATUS_ERROR;

    devid = GetDeviceID(deviceip);

    // If Device exists then read the data 
    if(devid >= 0)
    {
        GetParameter(deviceip, VNX_IPADDR);
        strcpy(respdata, lsw[devid].ipaddress);
    }
    else
        return STATUS_ERROR;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get Netmask from the device
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_GetNetmask(char* deviceip, char* respdata){
    int devid=0;
    if (!isDeviceSockOpen(deviceip))
        return STATUS_ERROR;

    devid = GetDeviceID(deviceip);

    // If Device exists then read the data 
    if(devid >= 0)
    {
        GetParameter(deviceip, VNX_IPMASK);
        strcpy(respdata, lsw[devid].netmask);
    }
    else
        return STATUS_ERROR;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get Gateway from the device
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_GetGateway(char* deviceip, char* respdata){
    int devid=0;
    if (!isDeviceSockOpen(deviceip))
        return STATUS_ERROR;

    devid = GetDeviceID(deviceip);

    // If Device exists then read the data 
    if(devid >= 0)
    {
        GetParameter(deviceip, VNX_IPGW);
        strcpy(respdata, lsw[devid].gateway);
    }
    else
        return STATUS_ERROR;
    return STATUS_OK;
}

//***************************************************************************
//
// Get the number of switches for this device - either 2, 4, or 8 for current products
//
//*****************************************************************************
STATUS_REPORT_T fnLSW_GetNumSwitches(char* deviceip, int* respdata)
{
    int devid = 0;
    if (!isDeviceSockOpen(deviceip))
        return STATUS_ERROR;

    devid = GetDeviceID(deviceip);

    // If Device exists then read the data
    if (devid >= 0) {
        *respdata = lsw[devid].NumSwitches;
    }
    else
        return STATUS_ERROR;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get max switch devices from the device
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_GetMaxSwitchDevices(char* deviceip, int* respdata){
    int devid=0;
    if (!isDeviceSockOpen(deviceip))
        return STATUS_ERROR;

    devid = GetDeviceID(deviceip);

    // If Device exists then read the data 
    if(devid >= 0)
    {
        GetParameter(deviceip, VNX_MAXPORTS);
        *respdata = lsw[devid].Maxswports / lsw[devid].NumSwitches;
    }
    else
        return STATUS_ERROR;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get current RF Switch state from the device
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_GetSwitchRFoutput(char* deviceip, int swindex, int* respdata){
    int ch, devid=0;
    if (!isDeviceSockOpen(deviceip))
        return STATUS_ERROR;

    devid = GetDeviceID(deviceip);

    // If Device exists then read the data 
    if(devid >= 0)
    {
    	if ((swindex >= 1) && (swindex <= lsw[devid].Maxswports / lsw[devid].NumSwitches))
    	{
        	GetParameterWithIndex(deviceip, VNX_SWSELECT, swindex);
        	*respdata = lsw[devid].rfswitchoutput[swindex-1];
    	}
		else
        {
			return STATUS_ERROR;
        }
    }
    else
        return STATUS_ERROR;
    return STATUS_OK;
}

//*****************************************************************************
//
// Function call to Set RF Switch output port of the corresponding switch index
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_SetSwitchRFoutput(char* deviceip, int swindex, LSW_SWPORT_T swport) {
    int devid=0;
    int swrfoutput;
	char *ptr;
	BYTE VNX_command[] = {0, 0, 0, 0, 0, 0, 0};

    devid = GetDeviceID(deviceip);

    if( devid >=0)
    {
        // Check Device open or not
        if (!isDeviceSockOpen(deviceip))
            return STATUS_ERROR;
		
		// Check the switch index
		if ((swindex >= 1) && (swindex <= lsw[devid].Maxswports / lsw[devid].NumSwitches) && (swport >= 1) && (swport <= lsw[devid].NumSwitches))
    	{
	        swrfoutput = lsw[devid].rfswitchoutput[swindex-1];
	        lsw[devid].rfswitchoutput[swindex-1] = swport;
			// RF Switch port
	        VNX_command[0] = (BYTE)(swport);
			// RF Switch index
		    VNX_command[4] = (BYTE)(swindex);
	        if (!SendReport(devid, VNX_SWSELECT | VNX_SET, (char *)VNX_command, 6)){
	            lsw[devid].rfswitchoutput[swindex-1] = swrfoutput;
	            return STATUS_ERROR;
	        }
	        return STATUS_OK;
		}
		return STATUS_ERROR;
    }
    else
        return STATUS_ERROR;
}

//*****************************************************************************
//
// Function call to save user settings to flash on device 
//  
//*****************************************************************************
STATUS_REPORT_T fnLSW_SaveSettings(char* deviceip) {
    int devid=0;
    devid = GetDeviceID(deviceip);
    if( devid >=0)
    {
        // Check Device open or not
        if (!isDeviceSockOpen(deviceip))
            return STATUS_ERROR;

        char VNX_savesettings[] = {0x42, 0x55, 0x31}; //three byte key to unlock the user protection.

        if (!SendReport(devid, VNX_SAVEPAR | VNX_SET, VNX_savesettings, 3))
            return STATUS_ERROR;
    }
    else
        return STATUS_ERROR;
  return STATUS_OK;
}
