// ---------------- Example program for the Vaunix RF Switch family ------------------
//
// -----------------------------------------------------------------------------------


#include <stdbool.h>	/* AK: Added include for 'bool' type */
#include <stdio.h>
#include <unistd.h>
#include "LSWhid.h"
#include <math.h>
#include <string.h>

#define FALSE 0
#define TRUE !FALSE

/* function prototypes */

/* globals */


/* code begins here */
int main (int argc, char *argv[]) {
  int nDevices, nActive;
  int i, j, result, status, devID;
  char cModelName[32];
  char c;
  char *username;
  DEVID activeDevices[MAXDEVICES] = {0};
  bool realhardware;
  char netbuff[MAX_NETBUFF];

  printf("Accessing USB ports on a Linux machine require root level\r\n");
  printf("access. If you are not logged in as root, you may be able to\r\n");
  printf("proceed if you have used 'chmod' to change the access mode\r\n");
  printf("of the appropriate devices in /dev/bus/usb. That requires\r\n");
  printf("root access also. We'll continue, but if you don't see your\r\n");
  printf("LSW devices or no data can be read from or written to them,\r\n");
  printf("that's probably the problem. su to root and try again.\r\n");
  printf("Try running with 'sudo', or become root by running 'su' before.\r\n\r\n");
    
  fnLSW_Init();
  /* If you have actual hardware attached, set this to TRUE. Setting to FALSE will run in test mode */
  realhardware = TRUE;
  fnLSW_SetTestMode(!realhardware);
  
  nDevices = fnLSW_GetNumDevices();
  printf("LSW test/demonstration program using library version %s\r\n\r\n", fnLSW_LibVersion());
  if (0 == nDevices) {
    printf("No Vaunix LSW devices located. Would you like to run in test mode? [Y/N]: "); fflush(0);
    c = getchar();
    if ('Y' == (c & 0xdf)) {
      printf("\r\nSwitching to test mode.\r\n");
      realhardware = FALSE;
      fnLSW_Init();
      fnLSW_SetTestMode(!realhardware);
      nDevices = fnLSW_GetNumDevices();
    }
  }

  if (nDevices == 0) {
    printf("No devices found. Exiting.\r\n");
    return 0;
  }

  // Get active device IDs
  nActive = fnLSW_GetDevInfo(activeDevices);

  printf("Found %d devices\r\n", nDevices);
  
  for (i = 0; i < sizeof(activeDevices)/sizeof(activeDevices[0]); i++) {
    devID = activeDevices[i];
    if (devID > 0) {
      result = fnLSW_GetModelName(devID, cModelName);
      printf("  Device %d is %s (%d chars)\r\n", devID, cModelName, result);
    }
  }
  printf("\r\n");

  printf("Opening devices...\r\n");
  for (i = 0; i < sizeof(activeDevices)/sizeof(activeDevices[0]); i++) {
    devID = activeDevices[i];
    if (devID > 0) {
      /* let's open and init each device to get the threads running */
      status = fnLSW_InitDevice(devID);
      printf("  Opened device %d. Status=0x%08x (%s)\r\n", devID, status, fnLSW_perror(status));
    }
  }

  /* the data structure is filled by polling and we need a few seconds to do that */
  
  for (i = 0; i < sizeof(activeDevices)/sizeof(activeDevices[0]); i++) {
    devID = activeDevices[i];
    if (devID > 0) {
      printf("\r\n");

      printf("Device %d is active\r\n", devID);
      status = fnLSW_GetModelName(devID, cModelName);
      printf("  Model: %s\r\n", cModelName);
      printf("  Serial Number: %d\r\n", fnLSW_GetSerialNumber(devID));

      if(strcmp(cModelName, "LSW-802P8T")==0 || 
        (strcmp(cModelName, "LSW-802P4T")==0) || 
        (strcmp(cModelName, "LSW-802PDT")==0) ||
        (strcmp(cModelName, "LSW-203PDT")==0) ||
        (strcmp(cModelName, "LSW-403PDT")==0))
      {
    
        printf("  GetMaxSwitchDevices returned: %d\r\n", fnLSW_GetMaxSwitchDevices(devID));
        printf("  GetNumSwitches returned: %d\r\n", fnLSW_GetNumSwitches(devID));
        
        printf("  GetIPMode returned: %d\r\n", fnLSW_GetIPMode(devID));
        fnLSW_GetIPAddress(devID, netbuff);
        printf("  GetIPAddress: %s\r\n", netbuff);
        fnLSW_GetNetmask(devID, netbuff);
        printf("  GetNetmask: %s\r\n", netbuff);
        fnLSW_GetGateway(devID, netbuff);
        printf("  GetGateway: %s\r\n", netbuff);

        for (j=1; j<=fnLSW_GetMaxSwitchDevices(devID); j++) {
          printf("  Switch %d RF Output: %d\r\n", j, fnLSW_GetSwitchRFoutput(devID, j));
        }

        for (j=1; j<=fnLSW_GetMaxSwitchDevices(devID); j++) {
          printf("Select switch %d RF output 1 for 3 seconds...\r\n", j);
          status = fnLSW_SetSwitchRFoutput(devID, j, 1);
          sleep(3);
          printf("  GetSwitchRFoutput returned: %d\r\n", fnLSW_GetSwitchRFoutput(devID, j));
          sleep(2);
          
          printf("Select switch %d RF output 2 for 3 seconds...\r\n", j);
          status = fnLSW_SetSwitchRFoutput(devID, j, 2);
          sleep(3);
          printf("  GetSwitchRFoutput returned: %d\r\n", fnLSW_GetSwitchRFoutput(devID, j));
          sleep(2);
        }

      }
      else
      {
        /* dump what we know - that we read from the hardware */
        if (realhardware) {
          printf("  GetNumSwitches returned %d\r\n", fnLSW_GetNumSwitches(devID));
          printf("  GetPulseOnTime returned %g\r\n", fnLSW_GetPulseOnTime(devID));
          printf("  GetPulseOffTime returned %g\r\n", fnLSW_GetPulseOffTime(devID));
          printf("  GetSwitchSetting returned %d\r\n", fnLSW_GetSwitchSetting(devID));
          printf("  GetActiveSwitch returned %d\r\n", fnLSW_GetActiveSwitch(devID));
        }
        
        for (j=1; j<=fnLSW_GetNumSwitches(devID); j++) {
          printf("Select switch %d for 3 seconds...\r\n", j);
          status = fnLSW_SetSwitch(devID, j);
          sleep(3);
          printf("  GetSwitchSetting returned %d\r\n", fnLSW_GetSwitchSetting(devID));
          sleep(2);
        }
        
        printf("Pulsed Mode for 3 seconds...\r\n");
        status = fnLSW_SetFastPulsedOutput(devID, .015, .03, TRUE);
        for (j=0; j<3*10; j++) {
          printf("  GetActiveSwitch returned %d\r\n", fnLSW_GetActiveSwitch(devID));
          usleep(100000);
        }
        status = fnLSW_EnableInternalPulseMod(devID, FALSE);
      }
    }
  }

  /* close the devices */
  printf("\nClosing devices...\r\n");
  for (i = 0; i < sizeof(activeDevices)/sizeof(activeDevices[0]); i++) {
    devID = activeDevices[i];
    if (devID > 0) {
      status = fnLSW_CloseDevice(devID);
      printf("  Closed device %d. Return status=0x%08x (%s)\r\n", devID, status, fnLSW_perror(status));
    }
  }
  printf("\nEnd of test\r\n");
  return 0;
}







