import ctypes
from time import sleep
import sys
from ctypes import c_int, c_float, create_string_buffer

# Load the shared library (adjust the path as needed)
vnx = ctypes.CDLL('./libLSWhid.so')

# Define constants
MAXDEVICES = 64
MAX_NETBUFF = 16

print("Accessing USB ports on a Linux machine require root level")
print("access. If you are not logged in as root, you may be able to")
print("proceed if you have used 'chmod' to change the access mode")
print("of the appropriate devices in /dev/bus/usb. That requires")
print("root access also. We'll continue, but if you don't see your")
print("LSW devices or no data can be read from or written to them,")
print("that's probably the problem. su to root and try again.")
print("Try running with 'sudo', or become root by running 'su' before.")

vnx.fnLSW_Init()

realhardware = True
vnx.fnLSW_SetTestMode(not realhardware)

n_devices = vnx.fnLSW_GetNumDevices()
vnx.fnLSW_LibVersion.restype = ctypes.c_char_p
print(f"\nLSW test/demonstration program using library version {vnx.fnLSW_LibVersion().decode('UTF-8')}\n")
if n_devices == 0:
    choice = input("No devices found. Run in test mode? [Y/N]: ")
    if choice.strip().upper() == 'Y':
        print("Switching to test mode.")
        realhardware = False
        vnx.fnLSW_Init()
        vnx.fnLSW_SetTestMode(True)
        n_devices = vnx.fnLSW_GetNumDevices()

if n_devices == 0:
    print("No devices found. Exiting.")
    sys.exit(0)


# Get active device IDs
active_devices = (c_int * MAXDEVICES)()
n_active = vnx.fnLSW_GetDevInfo(active_devices)

print(f"Found {n_devices} devices")

for dev_id in active_devices:
    if dev_id > 0:
        model_buf = create_string_buffer(32)
        result = vnx.fnLSW_GetModelName(dev_id, model_buf)
        print(f"  Device {dev_id} is {model_buf.value.decode()} ({result} chars)")
print()

print("Opening devices...")
vnx.fnLSW_perror.restype = ctypes.c_char_p
for dev_id in active_devices:
    if dev_id > 0:
        status = vnx.fnLSW_InitDevice(dev_id)
        print(f"  Opened device {dev_id}. Status: 0x{status:08x} ({vnx.fnLSW_perror(status).decode()})")

for dev_id in active_devices:
    if dev_id > 0:
        print()

        print(f"Device {dev_id} is active")
        model_buf = create_string_buffer(32)
        vnx.fnLSW_GetModelName(dev_id, model_buf)
        model_name = model_buf.value.decode()
        print(f"  Model: {model_name}")
        print(f"  Serial Number: {vnx.fnLSW_GetSerialNumber(dev_id)}")

        if model_name in ["LSW-802P8T", "LSW-802P4T", "LSW-802PDT", "LSW-203PDT", "LSW-403PDT"]:
            print(f"  GetMaxSwitchDevices returned: {vnx.fnLSW_GetMaxSwitchDevices(dev_id)}")
            print(f"  GetNumSwitches returned: {vnx.fnLSW_GetNumSwitches(dev_id)}")
            
            print(f"  GetIPMode returned: {vnx.fnLSW_GetIPMode(dev_id)}")
            ip_buf = create_string_buffer(MAX_NETBUFF)
            vnx.fnLSW_GetIPAddress(dev_id, ip_buf)
            print(f"  GetIPAddress: {ip_buf.value.decode()}")
            vnx.fnLSW_GetNetmask(dev_id, ip_buf)
            print(f"  GetNetmask: {ip_buf.value.decode()}")
            vnx.fnLSW_GetGateway(dev_id, ip_buf)
            print(f"  GetGateway: {ip_buf.value.decode()}")

            for i in range(1, vnx.fnLSW_GetMaxSwitchDevices(dev_id) + 1):
                print(f"  Switch {i} RF Output: {vnx.fnLSW_GetSwitchRFoutput(dev_id, i)}")

            for i in range(1, vnx.fnLSW_GetMaxSwitchDevices(dev_id) + 1):
                print(f"Select switch {i} RF output {1} for 3 seconds...")
                status = vnx.fnLSW_SetSwitchRFoutput(dev_id, i, 1)
                sleep(3)
                print(f"  GetSwitchRFoutput returned: {vnx.fnLSW_GetSwitchRFoutput(dev_id, i)}")
                sleep(2)

                print(f"Select switch {i} RF output {2} for 3 seconds...")
                status = vnx.fnLSW_SetSwitchRFoutput(dev_id, i, 2)
                sleep(3)
                print(f"  GetSwitchRFoutput returned: {vnx.fnLSW_GetSwitchRFoutput(dev_id, i)}")
                sleep(2)

        else:
            if (realhardware):
                print("  GetNumSwitches returned", vnx.fnLSW_GetNumSwitches(dev_id))
                vnx.fnLSW_GetPulseOnTime.restype = c_float
                print(f"  GetPulseOnTime returned {vnx.fnLSW_GetPulseOnTime(dev_id):.4f}")
                vnx.fnLSW_GetPulseOffTime.restype = c_float
                print(f"  GetPulseOffTime returned {vnx.fnLSW_GetPulseOffTime(dev_id):.4f}")
                print("  GetSwitchSetting returned", vnx.fnLSW_GetSwitchSetting(dev_id))
                print("  GetActiveSwitch returned", vnx.fnLSW_GetActiveSwitch(dev_id))
                
            for i in range(1, vnx.fnLSW_GetNumSwitches(dev_id) + 1):
                print(f"Select switch {i} for 3 seconds...")
                status = vnx.fnLSW_SetSwitch(dev_id, i)
                sleep(3)
                print("  GetSwitchSetting returned", vnx.fnLSW_GetSwitchSetting(dev_id))
                sleep(2)
            
            print("Pulsed Mode for 3 seconds...")
            status = vnx.fnLSW_SetFastPulsedOutput(dev_id, c_float(0.015), c_float(0.03), True)
            for _ in range(3*10):
                print("  GetActiveSwitch returned", vnx.fnLSW_GetActiveSwitch(dev_id))
                sleep(0.1)
            status = vnx.fnLSW_EnableInternalPulseMod(dev_id, False)

# Close devices
print("\nClosing devices...")
for dev_id in active_devices:
    if dev_id > 0:
        status = vnx.fnLSW_CloseDevice(dev_id)
        print(f"  Closed device {dev_id}. Status: 0x{status:08x} ({vnx.fnLSW_perror(status).decode()})")

print("\nEnd of test")
